const express = require('express');
const { authenticateToken, optionalAuthenticateToken } = require('../middleware/auth');
const { mysqlHelper } = require('../database/mysql-helper');
const crypto = require('crypto');
const CryptoJS = require('crypto-js');
const axios = require('axios');

// Import PhonePe SDK
const { StandardCheckoutClient, Env } = require('pg-sdk-node');

const router = express.Router();

// PhonePe Configuration using Official SDK
const PHONEPE_CONFIG = {
  clientId: process.env.PHONEPE_CLIENT_ID || 'TESTVVUAT_2502041721357207510164',
  clientSecret: process.env.PHONEPE_CLIENT_SECRET || 'ZTcxNDQyZjUtZjQ3Mi00MjJmLTgzOWYtMWZmZWQ2ZjdkMzVi',
  clientVersion: parseInt(process.env.PHONEPE_CLIENT_VERSION) || 1,
  merchantId: process.env.PHONEPE_MERCHANT_ID || 'TESTVVUAT',
  env: process.env.NODE_ENV === 'production' ? Env.PRODUCTION : Env.SANDBOX,
  redirectUrl: process.env.PHONEPE_REDIRECT_URL || 'http://localhost:3000/payment/success',
  callbackUrl: process.env.PHONEPE_CALLBACK_URL || 'http://192.168.0.11:3001/api/payments/phonepe/callback'
};

// Initialize PhonePe SDK Client
const phonePeClient = StandardCheckoutClient.getInstance(
  PHONEPE_CONFIG.clientId,
  PHONEPE_CONFIG.clientSecret,
  PHONEPE_CONFIG.clientVersion,
  PHONEPE_CONFIG.env
);

// Helper function to generate checksum using crypto-js
function generateChecksum(payload, endpoint) {
  const string = payload + endpoint + PHONEPE_CONFIG.clientSecret;
  const sha256 = CryptoJS.SHA256(string).toString();
  return sha256 + '###' + PHONEPE_CONFIG.keyIndex;
}

// Helper function to verify checksum
function verifyChecksum(receivedChecksum, payload, endpoint) {
  const expectedChecksum = generateChecksum(payload, endpoint);
  return receivedChecksum === expectedChecksum;
}

// Database helper functions
async function savePaymentTransaction(transactionId, orderId, status, amount = null, phonePeOrderId = null, phonePeTransactionId = null) {
  return new Promise((resolve, reject) => {
    mysqlHelper.run(
      `INSERT INTO payment_transactions (transaction_id, order_id, payment_method, status, amount, phonepe_order_id, phonepe_transaction_id, created_at) 
       VALUES (?, ?, 'phonepe', ?, ?, ?, ?, CURRENT_TIMESTAMP)`,
      [transactionId, orderId, status, amount, phonePeOrderId, phonePeTransactionId],
      (err, result) => {
        if (err) {
          console.error('Error saving payment transaction:', err);
          reject(err);
        } else {
          console.log(`Payment transaction ${transactionId} saved with PhonePe Order ID: ${phonePeOrderId}`);
          resolve(result);
        }
      }
    );
  });
}

async function updateOrderStatus(orderId, paymentStatus, transactionId, phonePeOrderId = null, phonePeTransactionId = null) {
  return new Promise((resolve, reject) => {
    // Map payment status to order status
    let orderStatus = 'pending';
    if (paymentStatus === 'completed') {
      orderStatus = 'confirmed';
      paymentStatus = 'paid'; // Use 'paid' as in your guide
    } else if (paymentStatus === 'failed') {
      orderStatus = 'cancelled';
    }

    mysqlHelper.run(
      `UPDATE orders 
       SET status = ?, payment_status = ?, transaction_id = ?, phonepe_order_id = ?, phonepe_transaction_id = ?, updated_at = CURRENT_TIMESTAMP
       WHERE id = ?`,
      [orderStatus, paymentStatus, transactionId, phonePeOrderId, phonePeTransactionId, orderId],
      (err, result) => {
        if (err) {
          console.error('Error updating order status:', err);
          reject(err);
        } else {
          console.log(`✅ Order ${orderId} status updated to ${orderStatus}, payment status: ${paymentStatus}`);
          console.log(`📱 PhonePe Order ID: ${phonePeOrderId}, Transaction ID: ${phonePeTransactionId}`);
          resolve(result);
        }
      }
    );
  });
}

// POST /api/payments/phonepe/initiate - Initiate PhonePe payment using Official SDK
router.post('/phonepe/initiate', optionalAuthenticateToken, async (req, res) => {
  try {
    // Handle both payload structures for flexibility
    const { paymentRequest: frontendPaymentRequest, orderData } = req.body;
    
    // Extract data from either structure
    const orderId = frontendPaymentRequest?.orderId || orderData?.orderId || req.body.orderId;
    const amount = frontendPaymentRequest?.amount || orderData?.amount || req.body.amount;
    const customerInfo = {
      name: frontendPaymentRequest?.customerName || orderData?.customerName || req.body.customerName,
      phone: frontendPaymentRequest?.customerPhone || orderData?.customerPhone || req.body.customerPhone,
      email: frontendPaymentRequest?.customerEmail || orderData?.customerEmail || req.body.customerEmail
    };

    if (!orderId || !amount) {
      return res.status(400).json({
        success: false,
        code: 'INVALID_REQUEST',
        message: 'Order ID and amount are required'
      });
    }

    // Generate unique merchant order ID (PhonePe requirement)
    const merchantOrderId = `ORDER_${orderId}_${Date.now()}`;
    
    // Validate order ID
    const validOrderId = parseInt(orderId);
    if (isNaN(validOrderId) || validOrderId > 2147483647) {
      return res.status(400).json({
        success: false,
        code: 'INVALID_ORDER_ID',
        message: 'Order ID must be a valid integer'
      });
    }

    console.log('PhonePe Payment Initiation Request:', { 
      orderData: req.body, 
      orderId, 
      amount,
      merchantOrderId 
    });

    // Check for mock mode
    const USE_MOCK = process.env.PHONEPE_USE_MOCK === 'true' || false;
    
    if (USE_MOCK) {
      console.log('🧪 Using MOCK PhonePe implementation for testing');
      
      await savePaymentTransaction(merchantOrderId, validOrderId, 'initiated', amount);
      
      const mockPaymentUrl = `http://192.168.0.11:3001/api/payments/phonepe/mock-payment?orderId=${validOrderId}&transactionId=${merchantOrderId}&amount=${amount}&customerName=${encodeURIComponent(customerInfo.name || 'Guest')}`;
      
      return res.json({
        success: true,
        code: 'PAYMENT_INITIATED',
        message: 'Payment initiated successfully (MOCK)',
        data: {
          merchantTransactionId: merchantOrderId,
          paymentUrl: mockPaymentUrl,
          merchantId: PHONEPE_CONFIG.merchantId,
          instrumentResponse: {
            type: 'PAY_PAGE',
            redirectInfo: {
              url: mockPaymentUrl,
              method: 'GET'
            }
          }
        }
      });
    }

    // REAL PhonePe implementation using Official SDK
    const { StandardCheckoutPayRequest } = require('pg-sdk-node');
    
    // Create redirect URL with transaction details
    const redirectUrl = `${PHONEPE_CONFIG.redirectUrl}?orderId=${validOrderId}&transactionId=${merchantOrderId}`;
    
    // Build payment request using official SDK
    const paymentRequest = StandardCheckoutPayRequest.builder()
      .merchantOrderId(merchantOrderId)
      .amount(Math.round(amount * 100)) // Convert to paise
      .redirectUrl(redirectUrl)
      .build();

    console.log('Making real PhonePe API call with SDK...');
    console.log('Payment Request:', {
      merchantOrderId,
      amount: Math.round(amount * 100),
      redirectUrl
    });

    // Save transaction details to database before API call
    await savePaymentTransaction(merchantOrderId, validOrderId, 'initiated', amount);

    // Make API call using PhonePe SDK
    const response = await phonePeClient.pay(paymentRequest);
    
    console.log('PhonePe SDK Response:', response);

    if (response && response.redirectUrl) {
      res.json({
        success: true,
        code: 'PAYMENT_INITIATED',
        message: 'Payment initiated successfully',
        data: {
          merchantTransactionId: merchantOrderId,
          paymentUrl: response.redirectUrl, // This will redirect to mercury-uat.phonepe.com
          merchantId: PHONEPE_CONFIG.merchantId,
          instrumentResponse: {
            type: 'PAY_PAGE',
            redirectInfo: {
              url: response.redirectUrl,
              method: 'GET'
            }
          }
        }
      });
    } else {
      throw new Error('Invalid response from PhonePe SDK');
    }

  } catch (error) {
    console.error('PhonePe initiate error:', error);
    res.status(500).json({
      success: false,
      code: 'PHONEPE_API_ERROR',
      message: `PhonePe API call failed: ${error.message}`,
      error: { message: error.message }
    });
  }
});

// GET /api/payments/phonepe/status/:transactionId - Check payment status
router.get('/phonepe/status/:transactionId', async (req, res) => {
  try {
    const { transactionId } = req.params;

    console.log('Checking PhonePe payment status for:', transactionId);

    // Check local database first
    const localTransaction = await new Promise((resolve, reject) => {
      mysqlHelper.get(
        'SELECT * FROM payment_transactions WHERE transaction_id = ?',
        [transactionId],
        (err, row) => {
          if (err) reject(err);
          else resolve(row);
        }
      );
    });

    if (!localTransaction) {
      return res.status(404).json({
        success: false,
        code: 'TRANSACTION_NOT_FOUND',
        message: 'Transaction not found'
      });
    }

    // FOR PRODUCTION: Use real PhonePe status check (set to true for mock testing)
    const USE_MOCK = process.env.PHONEPE_USE_MOCK === 'true' || false;
    
    if (USE_MOCK) {
      // Mock status response
      const orderId = transactionId.split('_')[1];
      
      const mockStatusResponse = {
        success: true,
        code: 'PAYMENT_SUCCESS',
        message: 'Transaction completed successfully',
        data: {
          merchantId: PHONEPE_CONFIG.merchantId,
          merchantTransactionId: transactionId,
          transactionId: `T${Date.now()}`,
          amount: Math.round(localTransaction.amount * 100),
          state: 'COMPLETED',
          responseCode: 'SUCCESS',
          paymentInstrument: { type: 'UPI' }
        }
      };
      
      // Update order status to "paid"
      await updateOrderStatus(orderId, 'completed', transactionId);
      
      console.log(`✅ Order ${orderId} marked as PAID (MOCK)`);
      
      res.json(mockStatusResponse);
      return;
    }
    
    // REAL PhonePe status check using SDK
    console.log('Making real PhonePe status check with SDK...');
    
    const response = await phonePeClient.getOrderStatus(transactionId);
    
    console.log('PhonePe Status Response:', response);

    if (response && response.state) {
      const orderId = transactionId.split('_')[1];
      
      // Extract PhonePe order ID and transaction ID from response
      const phonePeOrderId = response.orderId || null;
      const phonePeTransactionId = response.paymentDetails && response.paymentDetails.length > 0 
        ? response.paymentDetails[0].transactionId 
        : null;
      
      if (response.state === 'COMPLETED') {
        await updateOrderStatus(orderId, 'completed', transactionId, phonePeOrderId, phonePeTransactionId);
        console.log(`✅ Order ${orderId} marked as PAID`);
      } else if (response.state === 'FAILED') {
        await updateOrderStatus(orderId, 'failed', transactionId, phonePeOrderId, phonePeTransactionId);
        console.log(`❌ Order ${orderId} marked as FAILED`);
      }
      
      res.json({
        success: true,
        code: 'PAYMENT_STATUS_CHECKED',
        message: 'Status retrieved successfully',
        data: {
          ...response,
          internal_order_id: orderId,
          phonepe_order_id: phonePeOrderId,
          phonepe_transaction_id: phonePeTransactionId
        }
      });
    } else {
      res.json({
        success: false,
        code: 'STATUS_CHECK_FAILED',
        message: 'Could not retrieve payment status',
        data: response
      });
    }

  } catch (error) {
    console.error('Status check error:', error);
    res.status(500).json({
      success: false,
      code: 'PHONEPE_STATUS_ERROR',
      message: `PhonePe status check failed: ${error.response?.data?.message || error.message}`,
      error: error.response?.data || { message: error.message }
    });
  }
});

// Basic Auth middleware for webhook (uses env variables only)
const webhookAuth = (req, res, next) => {
  // Skip auth if webhook credentials are not configured
  if (!process.env.WEBHOOK_USERNAME || !process.env.WEBHOOK_PASSWORD) {
    return next();
  }

  const auth = req.headers.authorization;
  
  if (!auth || !auth.startsWith('Basic ')) {
    return res.status(401).json({
      success: false,
      message: 'Webhook authentication required'
    });
  }

  const credentials = Buffer.from(auth.split(' ')[1], 'base64').toString();
  const [username, password] = credentials.split(':');

  if (username === process.env.WEBHOOK_USERNAME && password === process.env.WEBHOOK_PASSWORD) {
    next();
  } else {
    res.status(401).json({
      success: false,
      message: 'Invalid webhook credentials'
    });
  }
};

// POST /api/payments/phonepe/callback - Handle PhonePe callback
router.post('/phonepe/callback', webhookAuth, async (req, res) => {
  try {
    console.log('PhonePe Callback received:', req.body);
    
    const { response: callbackResponse } = req.body;
    
    if (!callbackResponse) {
      return res.status(400).json({
        success: false,
        message: 'Invalid callback data'
      });
    }
    
    // Decode the response
    const decodedResponse = Buffer.from(callbackResponse, 'base64').toString();
    const paymentData = JSON.parse(decodedResponse);
    
    console.log('PhonePe Callback Data:', paymentData);
    
    if (paymentData.success && paymentData.data) {
      const transactionData = paymentData.data;
      const orderId = transactionData.merchantTransactionId.split('_')[1];
      
      if (transactionData.state === 'COMPLETED') {
        await updateOrderStatus(orderId, 'completed', transactionData.transactionId || transactionData.merchantTransactionId);
        console.log(`✅ Order ${orderId} marked as PAID via callback`);
      } else if (transactionData.state === 'FAILED') {
        await updateOrderStatus(orderId, 'failed', transactionData.transactionId || transactionData.merchantTransactionId);
        console.log(`❌ Order ${orderId} marked as FAILED via callback`);
      }
    }
    
    res.json({ success: true });
    
  } catch (error) {
    console.error('PhonePe callback processing error:', error);
    res.status(500).json({
      success: false,
      message: 'Callback processing failed'
    });
  }
});

// GET /api/payments/phonepe/test - Test endpoint for development
router.get('/test', (req, res) => {
  res.json({
    success: true,
    message: 'PhonePe payment routes are working',
    config: {
      merchantId: PHONEPE_CONFIG.merchantId,
      apiUrl: PHONEPE_CONFIG.apiUrl,
      environment: process.env.NODE_ENV || 'development'
    }
  });
});

// POST /api/payments/phonepe/complete-test - Test endpoint to simulate payment completion
router.post('/phonepe/complete-test', async (req, res) => {
  try {
    const { transactionId, orderId } = req.body;
    
    if (!transactionId && !orderId) {
      return res.status(400).json({
        success: false,
        message: 'Transaction ID or Order ID is required'
      });
    }
    
    console.log(`🧪 Simulating payment completion for transaction: ${transactionId}, order: ${orderId}`);
    
    // Find the transaction
    let transaction = null;
    if (transactionId) {
      transaction = await new Promise((resolve, reject) => {
        mysqlHelper.get(
          'SELECT * FROM payment_transactions WHERE transaction_id = ?',
          [transactionId],
          (err, row) => {
            if (err) reject(err);
            else resolve(row);
          }
        );
      });
    } else if (orderId) {
      transaction = await new Promise((resolve, reject) => {
        mysqlHelper.get(
          'SELECT * FROM payment_transactions WHERE order_id = ? ORDER BY created_at DESC LIMIT 1',
          [orderId],
          (err, row) => {
            if (err) reject(err);
            else resolve(row);
          }
        );
      });
    }
    
    if (!transaction) {
      return res.status(404).json({
        success: false,
        message: 'Transaction not found'
      });
    }
    
    // Update payment transaction to completed
    await new Promise((resolve, reject) => {
      mysqlHelper.run(
        'UPDATE payment_transactions SET status = ?, phonepe_response = ?, updated_at = CURRENT_TIMESTAMP WHERE transaction_id = ?',
        ['completed', JSON.stringify({state: 'COMPLETED', test: true}), transaction.transaction_id],
        (err, result) => {
          if (err) reject(err);
          else resolve(result);
        }
      );
    });
    
    // Update order status to confirmed and payment status to completed
    await new Promise((resolve, reject) => {
      mysqlHelper.run(
        'UPDATE orders SET status = ?, payment_status = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?',
        ['confirmed', 'completed', transaction.order_id],
        (err, result) => {
          if (err) reject(err);
          else resolve(result);
        }
      );
    });
    
    console.log(`✅ Order ${transaction.order_id} marked as PAID and CONFIRMED`);
    
    res.json({
      success: true,
      message: 'Payment completed successfully',
      data: {
        orderId: transaction.order_id,
        transactionId: transaction.transaction_id,
        status: 'completed'
      }
    });
    
  } catch (error) {
    console.error('Payment completion test error:', error);
    res.status(500).json({
      success: false,
      message: 'Internal server error'
    });
  }
});

// POST /api/payments/phonepe/simulate-callback - Simulate PhonePe callback for testing
router.post('/phonepe/simulate-callback', async (req, res) => {
  try {
    const { transactionId, orderId, status = 'COMPLETED' } = req.body;
    
    if (!transactionId && !orderId) {
      return res.status(400).json({
        success: false,
        message: 'Transaction ID or Order ID is required'
      });
    }
    
    console.log(`🧪 Simulating PhonePe callback for transaction: ${transactionId}, order: ${orderId}, status: ${status}`);
    
    // Find the transaction
    let transaction = null;
    if (transactionId) {
      transaction = await new Promise((resolve, reject) => {
        mysqlHelper.get(
          'SELECT * FROM payment_transactions WHERE transaction_id = ?',
          [transactionId],
          (err, row) => {
            if (err) reject(err);
            else resolve(row);
          }
        );
      });
    } else if (orderId) {
      transaction = await new Promise((resolve, reject) => {
        mysqlHelper.get(
          'SELECT * FROM payment_transactions WHERE order_id = ? ORDER BY created_at DESC LIMIT 1',
          [orderId],
          (err, row) => {
            if (err) reject(err);
            else resolve(row);
          }
        );
      });
    }
    
    if (!transaction) {
      return res.status(404).json({
        success: false,
        message: 'Transaction not found'
      });
    }
    
    // Simulate PhonePe callback data
    const callbackData = {
      success: true,
      code: 'PAYMENT_SUCCESS',
      message: 'Your payment is successful.',
      data: {
        merchantId: PHONEPE_CONFIG.merchantId,
        merchantTransactionId: transaction.transaction_id,
        transactionId: `PH_${Date.now()}`,
        amount: Math.round(transaction.amount * 100),
        state: status,
        responseCode: 'SUCCESS',
        paymentInstrument: {
          type: 'UPI'
        }
      }
    };
    
    // Update payment transaction
    await new Promise((resolve, reject) => {
      mysqlHelper.run(
        'UPDATE payment_transactions SET status = ?, phonepe_response = ?, updated_at = CURRENT_TIMESTAMP WHERE transaction_id = ?',
        [status.toLowerCase(), JSON.stringify(callbackData), transaction.transaction_id],
        (err, result) => {
          if (err) reject(err);
          else resolve(result);
        }
      );
    });
    
    // Update order status based on payment status
    let orderStatus = 'pending';
    let paymentStatus = 'pending';
    
    if (status === 'COMPLETED') {
      orderStatus = 'confirmed';
      paymentStatus = 'completed';
    } else if (status === 'FAILED') {
      orderStatus = 'cancelled';
      paymentStatus = 'failed';
    }
    
    await new Promise((resolve, reject) => {
      mysqlHelper.run(
        'UPDATE orders SET status = ?, payment_status = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?',
        [orderStatus, paymentStatus, transaction.order_id],
        (err, result) => {
          if (err) reject(err);
          else resolve(result);
        }
      );
    });
    
    console.log(`✅ Order ${transaction.order_id} updated: status=${orderStatus}, payment_status=${paymentStatus}`);
    
    res.json({
      success: true,
      message: 'Callback simulation completed',
      data: {
        orderId: transaction.order_id,
        transactionId: transaction.transaction_id,
        status: paymentStatus,
        orderStatus: orderStatus
      }
    });
    
  } catch (error) {
    console.error('Callback simulation error:', error);
    res.status(500).json({
      success: false,
      message: 'Internal server error'
    });
  }
});

// GET /api/payments/phonepe/mock-payment - Mock payment page for testing
router.get('/phonepe/mock-payment', (req, res) => {
  const { orderId, transactionId, amount, customerName } = req.query;
  
  const mockPaymentPage = `
  <!DOCTYPE html>
  <html>
  <head>
    <title>PhonePe - Test Payment</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <style>
      body {
        font-family: Arial, sans-serif;
        background: linear-gradient(135deg, #5f259f 0%, #7b68ee 100%);
        min-height: 100vh;
        display: flex;
        align-items: center;
        justify-content: center;
        margin: 0;
        padding: 20px;
      }
      .payment-container {
        background: white;
        border-radius: 16px;
        padding: 40px;
        box-shadow: 0 25px 50px rgba(95, 37, 159, 0.15);
        max-width: 450px;
        width: 100%;
        text-align: center;
      }
      .phonepe-logo {
        width: 80px;
        height: 80px;
        background: linear-gradient(135deg, #5f259f, #7b68ee);
        border-radius: 20px;
        margin: 0 auto 20px;
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        font-weight: bold;
        font-size: 28px;
      }
      .amount {
        font-size: 36px;
        font-weight: bold;
        color: #2d3436;
        margin: 20px 0;
      }
      .order-info {
        background: #f8f9fa;
        border-radius: 12px;
        padding: 20px;
        margin: 25px 0;
        text-align: left;
      }
      .payment-btn {
        display: block;
        width: 100%;
        padding: 16px 24px;
        margin: 12px 0;
        border: none;
        border-radius: 12px;
        font-size: 16px;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.3s ease;
      }
      .primary {
        background: linear-gradient(135deg, #5f259f, #7b68ee);
        color: white;
      }
      .secondary {
        background: white;
        color: #e74c3c;
        border: 2px solid #e74c3c;
      }
      .test-badge {
        position: absolute;
        top: 15px;
        right: 15px;
        background: #e74c3c;
        color: white;
        padding: 8px 16px;
        border-radius: 20px;
        font-size: 12px;
        font-weight: bold;
      }
    </style>
  </head>
  <body>
    <div class="test-badge">TEST MODE</div>
    <div class="payment-container">
      <div class="phonepe-logo">Pe</div>
      <h1>PhonePe Payment</h1>
      <div class="amount">₹${amount}</div>
      
      <div class="order-info">
        <h4>Order Details</h4>
        <p><strong>Order ID:</strong> #${orderId}</p>
        <p><strong>Transaction ID:</strong> ${transactionId.substring(0, 20)}...</p>
        <p><strong>Customer:</strong> ${customerName || 'Guest'}</p>
        <p><strong>Merchant:</strong> USASYA ECO SOLUTIONS</p>
      </div>
      
      <button class="payment-btn primary" onclick="simulatePayment('success')">
        ✅ Complete Payment Successfully
      </button>
      <button class="payment-btn secondary" onclick="simulatePayment('failed')">
        ❌ Simulate Payment Failure
      </button>
      
      <p style="font-size: 12px; color: #636e72; margin-top: 20px;">
        This is a test environment. No real money will be charged.
      </p>
    </div>
    
    <script>
      function simulatePayment(status) {
        if (status === 'success') {
          alert('🎉 Payment Successful!\\nRedirecting to success page...');
          window.location.href = 'http://localhost:3000/payment/success?orderId=${orderId}&transactionId=${transactionId}&status=success&amount=${amount}';
        } else {
          alert('❌ Payment Failed!\\nRedirecting back...');
          window.location.href = 'http://localhost:3000/payment/success?orderId=${orderId}&transactionId=${transactionId}&status=failed&amount=${amount}';
        }
      }
    </script>
  </body>
  </html>
  `;
  
  res.send(mockPaymentPage);
});

module.exports = router;
